<?php

namespace App\Services;

use App\Models\STOCK\Stocksnapshot;
use Illuminate\Support\Facades\DB;

class StockSnapshotService
{


    /**
     * Apply stock mutation based on operation type
     *
     * @param int         $productId
     * @param string      $operationType  entrynote | exitenote | adjustmentnote
     * @param int         $quantity
     * @param float|null  $price          selling price
     * @param float|null  $cost           cost price
     */
    public static function apply(
        int $productId,
        string $operationType,
        int $quantity,
        ?float $price = null,
        ?float $cost = null
    ): void {
        DB::transaction(function () use (
            $productId,
            $operationType,
            $quantity,
            $price,
            $cost
        ) {
            $handleStockWithInOut = filter_var(
                env('HANDLE_STOCK_WITH_IN_OUT', false),
                FILTER_VALIDATE_BOOL
            );

            // lock row or create snapshot
            $snapshot = Stocksnapshot::where('productId', $productId)
                ->lockForUpdate()
                ->first();

            if (!$snapshot) {
                $snapshot = Stocksnapshot::create([
                    'productId' => $productId,
                    'stock'     => 0,
                    'lastPrice' => 0,
                    'lastCost'  => 0,
                ]);
            }

            // Apply stock logic
            if ($handleStockWithInOut) {
                switch ($operationType) {
                    case 'receiptnote':
                        $snapshot->stock += $quantity;
                        break;

                    case 'deliverynote':
                        $snapshot->stock -= $quantity;
                        break;

                    case 'adjustmentnote':
                        // adjustment = set absolute stock
                        $snapshot->stock = $quantity;
                        break;
                }
            } else {
                switch ($operationType) {
                    case 'entrynote':
                        $snapshot->stock += $quantity;
                        break;

                    case 'exitnote':
                        $snapshot->stock -= $quantity;
                        break;

                    case 'adjustmentnote':
                        // adjustment = set absolute stock
                        $snapshot->stock = $quantity;
                        break;
                }
            }
            $updatePrice = filter_var(
                env('UPDATE_PRICE_AFTER_OPERATION', false),
                FILTER_VALIDATE_BOOL
            );
            if ($updatePrice) {
                // Update prices if provided
                if (!is_null($price)) {
                    $snapshot->lastPrice = $price;
                }

                if (!is_null($cost)) {
                    $snapshot->lastCost = $cost;
                }
            }


            $snapshot->save();
        });
    }

    //remove case
    public static function rollback(
        int $productId,
        string $operationType,
        int $quantity
    ): void {
        DB::transaction(function () use ($productId, $operationType, $quantity) {

            $handleStockWithInOut = filter_var(
                env('HANDLE_STOCK_WITH_IN_OUT', false),
                FILTER_VALIDATE_BOOL
            );

            $snapshot = Stocksnapshot::where('productId', $productId)
                ->lockForUpdate()
                ->first();

            if (!$snapshot) {
                return;
            }

            if ($handleStockWithInOut) {

                switch ($operationType) {

                    case 'receiptnote':
                        // reverse of +=
                        $snapshot->stock -= $quantity;
                        break;

                    case 'deliverynote':
                        // reverse of -=
                        $snapshot->stock += $quantity;
                        break;

                    case 'adjustmentnote':
                        // safest solution: force recalculation later
                        break;
                }
            } else {

                switch ($operationType) {

                    case 'entrynote':
                        $snapshot->stock -= $quantity;
                        break;

                    case 'exitnote':
                        $snapshot->stock += $quantity;
                        break;

                    case 'adjustmentnote':
                        break;
                }
            }

            $snapshot->save();
        });
    }
}
