<?php

namespace App\Http\Controllers\OPERATION;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Models\OPERATION\Operation;
use App\Models\admin\Setting;
use Illuminate\Http\Request;
use App\Traits\CrudTrait;
use App\Services\GenerateInvoice;
use Illuminate\Support\Facades\Mail;
use App\Mail\GeneralNotificationMail;


class OperationController extends Controller
{
    use CrudTrait;

    protected string $modelClass = Operation::class;
    protected string $dataType = 'operation';
    public function index(Request $request)
    {
        return $this->getData($request);
    }

    public function store(Request $request)
    {
        return $this->storeData($request);
    }

    public function update(Request $request)
    {
        return $this->updateData($request);
    }

    public function bulkUpdate(Request $request)
    {
        return $this->multiUpdate($request);
    }

    public function destroy(Request $request)
    {
        return $this->destroyData($request);
    }

    public function bulkDestory(Request $request)
    {
        return $this->bulkDelete($request);
    }

    public function clone(Request $request)
    {
        return $this->cloneData($request);
    }

    public function export(Request $request)
    {
        return $this->exportData($request);
    }

    public function import(Request $request)
    {
        return $this->importData($request);
    }

    public function upload(Request $request)
    {
        return $this->uploadFile($request);
    }

    public function download(Request $request)
    {
        return $this->downloadFile($request);
    }

    //--------------Helpers------------------------------------------------------


    public function generateInvoiceNumber(Request $request)
    {
        $docType = $request->input('docType');
        $code = $this->docTypeCodes[$docType] ?? strtoupper($docType);

        $pattern = env('DOCID_PATTERN', '6');

        preg_match('/(\d+)/', $pattern, $padMatch);
        $pad = isset($padMatch[1]) ? (int) $padMatch[1] : 6;

        preg_match('/^\D/', $pattern, $sepMatch);
        $separator = $sepMatch[0] ?? '';

        $counter = DB::table('operation_counters')
            ->where('docType', $docType)
            ->first();

        $nextNumber = $counter ? $counter->last_number + 1 : 1;
        $formattedNumber = str_pad($nextNumber, $pad, '0', STR_PAD_LEFT);

        $date = env('DOCID_PATTERN_HAS_DATE') ? now()->format('Ymd'). $separator : '';
        $invoiceNumber = $code . $separator . $date  . $formattedNumber;


        return response()->json([
            'invoiceNumber' => $invoiceNumber
        ]);
    }



    public function createOperationCounter(Request $request)
    {
        $docType = $request->input('docType');

        DB::transaction(function () use ($docType) {
            $counter = DB::table('operation_counters')
                ->where('docType', $docType)
                ->lockForUpdate()
                ->first();

            if (!$counter) {
                DB::table('operation_counters')->insert([
                    'docType' => $docType,
                    'last_number' => 1,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);

                $nextNumber = 1;
            } else {
                $nextNumber = $counter->last_number + 1;
                DB::table('operation_counters')
                    ->where('docType', $docType)
                    ->update(['last_number' => $nextNumber, 'updated_at' => now()]);
            }

            return $nextNumber;
        });
    }



    protected $docTypeCodes = [
        //sales
        'invoicesale'       => 'FA',
        'returnsale'        => 'AVV',
        'estimate'          => 'DEV',
        'deliverynote'      => 'BL',
        'possale'           => 'POS',
        //purchases
        'pricerequest'      => 'DP',
        'purchaseorder'     => 'BC',
        'receiptnote'       => 'BR',
        'invoicepurchase'   => 'FA',
        'returnpurchase'    => 'AVA',
        'expense'           => 'DEP',
        //stock movements
        'exitnote'          => 'BS',
        'entrynote'         => 'BE',
        'transfernote'      => 'BT',
        'adjustmentnote'    => 'BAJ',
        //production
        'orderprod'         => 'OP',
        'generateprod'      => 'BP',

    ];


    public function sendBulkMail(Request $request, GenerateInvoice $pdfService)
    {
        $mailData = Setting::query()->first();

        // Dynamic mail config
        config([
            'mail.default' => 'dynamic',
            'mail.mailers.dynamic' => [
                'transport'  => 'smtp',
                'host'       => $mailData->company['host'],
                'port'       => (int) $mailData->company['port'],
                'encryption' => $mailData->company['encryption'],
                'username'   => $mailData->company['username'],
                'password'   => $mailData->company['password'],
            ],
            'mail.from.address' => $mailData->company['fromAddress'] ?? $mailData->company['username'],
            'mail.from.name'    => $mailData->company['fromName'] ?? config('app.name'),
        ]);

        $recipients = $request->sendTo ?? $request->counterParty->email;

        if (empty($recipients)) {
            $counterParty = $request->counterParty ?? null;
            if (!$counterParty || empty($counterParty['email'])) {
                return response()->json(['error' => 'No email found'], 400);
            }
            $recipients[] = $counterParty;
        }

        // 🧾 Generate PDF once
        $pdfContent = null;

        if ($request->docId) {
            $operation = Operation::where('docId', $request->docId)->first();
            if ($operation) {
                $pdf = $pdfService->generate($operation, $mailData->company, $mailData->operation);
                $pdfContent = $pdf->output(); // 👈 PDF as string
            }
        }

        // 🚀 Send after response (shared hosting friendly)
        app()->terminating(function () use ($recipients, $request, $pdfContent) {
            foreach ($recipients as $recipient) {
                if (empty($recipient['email'])) {
                    continue;
                }

                Mail::to($recipient['email'])->send(
                    new GeneralNotificationMail(
                        $request->subject,
                        $request->body,
                        $request->docId,
                        $request->docType ?? null,
                        $pdfContent // 📎
                    )
                );
            }
        });

        return response()->json([
            'success' => 'Emails sent with PDF attached'
        ]);
    }




    //print
    public function downloadInvoice(Request $request, GenerateInvoice $pdfService)
    {
        $operation = Operation::query()->where('docId', $request->docId)->first();
        $setting = Setting::query()->first();
        $pdf = $pdfService->generate($operation, $setting->company, $setting->operation);
        Log::info('Generating PDF for operation: ', ['operationSetting' => $setting->operation['hasBackgroundLogo']]);
        //rename the  pdf
        return $pdf->download(
            $operation->docId . '_' . $operation->counterParty['name'] . '.pdf'
        );
    }
}
