<?php

namespace App\Http\Controllers\HRM;

use App\Http\Controllers\Controller;
use App\Traits\CrudTrait;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Rmunate\Utilities\SpellNumber;
use App\Models\HRM\Payroll;
use App\Models\HRM\Leave;
use App\Models\HRM\Compensation;
use App\Models\HRM\Advance;
use App\Models\HRM\Absence;
use App\Models\HRM\Attendance;
use App\Models\HRM\Employee;

class PayrollController extends Controller
{
    use CrudTrait;

    private const SUPPLEMENTARY_TYPES = [
        'suppDay' => [
            'label' => 'Heures supplémentaires de jour',
            'rate' => 1.25,
            'rateLabel' => '+25%',
        ],
        'suppNight' => [
            'label' => 'Heures supplémentaires de nuit',
            'rate' => 1.5,
            'rateLabel' => '+50%',
        ],
        'suppHoly' => [
            'label' => 'Heures jours fériés',
            'rate' => 2.0,
            'rateLabel' => '+100%',
        ],
        'weekend' => [
            'label' => 'Heures week-end',
            'rate' => 1.5,
            'rateLabel' => '+50%',
        ],
    ];

    /**
     * 📊 Complete Social Security Constants 2025 (Employer + Employee)
     */
    // CNSS - Caisse Nationale de Sécurité Sociale
    private const CNSS_EMPLOYEE_RATE = 0.0429;  // 4.29% salarié  ?

    private const CNSS_EMPLOYER_RATE = 0.086;  // 8.6% employeur ?
    private const CNSS_TOTAL_RATE = 0.1289;  // 12.89% total   ?
    private const CNSS_CAP = 6000;  // Plafond 6,000 DH
    private const CNSS_RATE = 0.0448;  // 4.48% employee share 2025
    private const CNSS_FLOOR = 3266.1;  // SMIG 2025 comme plancher
    // Allocations Familiales
    private const FAMILY_ALLOWANCE_RATE = 0.064;  // 6.40% employeur
    private const FAMILY_ALLOWANCE_CAP = 6000;  // Plafond identique à CNSS
    // AMO - Assurance Maladie Obligatoire
    private const AMO_EMPLOYEE_RATE = 0.0226;  // 2.26% salarié????
    private const AMO_EMPLOYER_RATE = 0.0226;  // 2.26% + 1.85 employeur
    private const AMO_TOTAL_RATE = 0.0637;  // 4.52% total + 1.85% employeur = 6.37%
    private const AMO_CAP = null;  // Aucun plafond
    private const AMO_RATE = 0.0231;  // Plafond aligné sur CNSS
    private const AMO_FLOOR = 3266.1;  // SMIG 2025
    // IPE - Indemnité de Perte d'Emploi
    private const IPE_EMPLOYEE_RATE = 0.0019;  // 0.19% salarié
    private const IPE_EMPLOYER_RATE = 0.0038;  // 0.38% employeur
    private const IPE_TOTAL_RATE = 0.0057;  // 0.57% total
    private const IPE_CAP = 6000;  // Plafond 6,000 DH
    // TFP - Taxe de Formation Professionnelle
    private const TFP_EMPLOYEE_RATE = 0.0;  // 0% salarié
    private const TFP_EMPLOYER_RATE = 0.016;  // 1.6% employeur
    private const TFP_TOTAL_RATE = 0.016;  // 1.6% total
    private const TFP_CAP = null;  // Aucun plafond
    private const TAX_CREDIT_MAX = 250;
    private const USE_BASE_SALARY = false;
    /**
     * 📊 SMIG 2025 Constants (Salaire Minimum Interprofessionnel Garanti)
     */
    private const SMIG_HOURLY = 17.1;  // 17.10 DH/heure

    private const SMIG_MONTHLY_HOURS = 191;  // Heures mensuelles
    private const SMIG_MONTHLY = 3266.1;  // 17.10 × 191 = 3,266.10 DH
    private const LOSS_JOB_RATE = 0.0021;  // 0.21% en 2025
    private const LOSS_JOB_CAP = 7000;  // Plafond aligné sur CNSS
    private const LOSS_JOB_FLOOR = 3266.1;  // SMIG 2025
    private const PRO_FEES_RATE_LOW = 0.35;  // 35% for gross <= 6500 DH
    private const PRO_FEES_RATE_HIGH = 0.25;  // 25% for gross > 6500 DH
    private const PRO_FEES_THRESHOLD = 6500;  // Threshold amount
    private const PRO_FEES_CAP_LOW = 2916.66;  // DH max deduction
    private const PRO_FEES_CAP_HIGH = 2500;  // DH max deduction

    /**
     * 📊 IR Brackets (EXACT from Excel formula)
     */
    private const IR_BRACKETS = [
        ['max' => 3333.33, 'rate' => 0.0, 'deduction' => 0],
        ['max' => 5001.0, 'rate' => 0.1, 'deduction' => 333.33],
        ['max' => 6666.67, 'rate' => 0.2, 'deduction' => 833.33],
        ['max' => 8333.33, 'rate' => 0.3, 'deduction' => 1500],
        ['max' => 15000.0, 'rate' => 0.34, 'deduction' => 1833.33],
        ['max' => null, 'rate' => 0.37, 'deduction' => 2283.33],
    ];

    // Tax credit per dependent
    private const TAX_CREDIT_PER_DEPENDENT = 41.67;

    /** 📊 Exemption Limits for Allowances (DGI Morocco) */

    /**
     * 📊 Family Charge Settings (Moroccan Law)
     */
    private const FAMILY_CHARGE_PER_CHILD = 41.67;

    private const FAMILY_CHARGE_MARRIED_NO_CHILDREN = 41.67;
    private const FAMILY_CHARGE_MAX = 180;

    private const EXEMPTION_LIMITS = [
        'transport' => 500,  // Transport allowance monthly exemption
        'meal' => 300,  // Meal allowance monthly exemption
        'family' => 300,  // Family allowance per child
        'housing' => 2500,  // Housing allowance annual limit divided by 12
    ];

    private const SUPP_RATES = [
        'regular' => 1.0,  // workedRegularHours (already counted in base salary)
        'day' => 1.25,  // workedSuppDayHours
        'night' => 1.5,  // workedSuppNightHours
        'holy' => 2.0,  // workedSuppHolyDaysHours
        'weekend' => 1.5,  // workedWeekendHours
    ];

    /**
     * 📊 Retirement Constants (CNSS Morocco)
     */
    private const RETIREMENT_MIN_DAYS = 3240;  // 3240 jours minimum

    private const RETIREMENT_BASE_RATE = 0.5;  // 50% taux de base
    private const RETIREMENT_BONUS_DAYS = 216;  // 216 jours pour 1% supplémentaire
    private const RETIREMENT_BONUS_RATE = 0.01;  // 1% par tranche
    private const RETIREMENT_MAX_RATE = 0.7;  // 70% taux maximum
    private const RETIREMENT_AVERAGE_MONTHS = 96;  // 96 mois pour calcul moyenne

    protected string $modelClass = Payroll::class;
    protected string $dataType = 'object';

    // =========================================================
    // 🔹 Base Routes
    // =========================================================



    // =========================================================
    // 🔹 Payroll Generation with Rate Storage
    // =========================================================

    public function store(Request $request)
    {
        $date = Carbon::parse($request->date);
        $month = $date->month;
        $year = $date->year;
        $requestedMonth = $date->copy()->startOfMonth();
        $currentMonth = now()->startOfMonth();

        if ($requestedMonth->gt($currentMonth)) {
            return response()->json([
                'error' => 'You can not generate payroll for future month'
            ], 400);
        }

        $employees = $request->input('employee');

        if (is_string($employees)) {
            $employees = json_decode($employees, true);
        }
        if (!is_array($employees)) {
            return response()->json(['error' => 'Invalid employees format'], 400);
        }

        $payrolls = [];

        foreach ($employees as $empData) {

            $employee = (object) $empData;
            $employeeId = $employee->id;
            $existingPayroll = Payroll::withTrashed()
                ->where('employee_id',  $employee->id)
                ->where('month', $date->format('Y-m'))
                ->whereNull('deleted_at')
                ->first();

            if ($existingPayroll) {
                continue;
            }
            $attendance = $this->getAttendanceSummary($employeeId, $month, $year);
            $absences = $this->getUnjustifiedAbsenceDays($employeeId, $month, $year);
            $absencesJustified = $this->getAbsencesJustified($employeeId, $month, $year);
            $leaves = $this->getLeaves($employeeId, $month, $year);
            $advances = $this->getAdvances($employeeId, $month, $year);
            $compensations = $this->getCompensations($employeeId, $month, $year);
            $daysDeclared = (float) ($employee->daysDeclared ?? 0);
            $workingHours = (float) ($employee->workingHours ?? 0);
            $salary       = (float) ($employee->salary ?? 0);

            $hourRegularRate = ($daysDeclared > 0 && $workingHours > 0)
                ? $salary / $daysDeclared / $workingHours
                : 17.10;

            $detailSupp = $this->detailSupplementaryPay($attendance, $hourRegularRate);

            $salaryCalculation = $this->calculateGrossSalary($employee, $attendance, $compensations, $absences, $absencesJustified, $leaves,  $month, $year);

            $grossSalary = $salaryCalculation['gross_salary'];
            $cnssBase = $salaryCalculation['cnss_base'];
            $taxableCompensations = $salaryCalculation['taxable_compensations'];
            $nonTaxableCompensations = $salaryCalculation['non_taxable_compensations'];

            // Get employee family data for family allowance
            $familyData = $this->getEmployeeFamilyData($employeeId);
            $childrenCount = $familyData['children_count'];

            // Calculate total worked hours for SMIG validation
            $totalWorkedHours = $attendance['workedRegular']
                + $attendance['suppDay']
                + $attendance['suppNight']
                + $attendance['suppHoly']
                + $attendance['weekend'];

            // Validate SMIG compliance
            $smigValidation = $this->validateSMIGCompliance($grossSalary, $totalWorkedHours);

            // 🔥 UPDATED: Calculate complete social security with family allowance
            $socialSecurity = $this->calculateCompleteSocialSecurity($cnssBase, $salaryCalculation['gross_salary_taxable'], $childrenCount);

            // Professional Fees
            $proFeesCalculation = $this->calculateProFeesWithRate($salaryCalculation['gross_salary_taxable']);

            // Retirement
            $retirementCalculation = $this->calculateRetirementWithRate($employeeId, $grossSalary);

            $totalDeductions = $socialSecurity['employee']['total_amount'] + $proFeesCalculation['amount'];

            // Calculate IR
            $irCalculation = $this->calculateIRWithRate(
                $salaryCalculation['gross_salary_taxable'],
                $totalDeductions,
                $employeeId
            );


            // Employee total deductions (for net salary)
            $totalEmployeeDeductions = $socialSecurity['employee']['total_amount']
                + $irCalculation['amount'];

            $netSalary = $grossSalary - $totalEmployeeDeductions - $advances;

            $payrolls[] = [
                'uid' => (string) Str::uuid(), //employee
                'name' => $employee->name, //employee
                'employee_id' => $employeeId, //employee
                'employeeData' => json_encode($employee), //employee
                'status' => 'pending', //payroll
                'date' => $date->toDateString(), //input
                'month' => $date->format('Y-m'), //input
                // Hours and attendance
                'totalWorkedRegularHours' => $attendance['workedRegular'], //input
                'totalWorkedSuppDayHours' => $attendance['suppDay'], //input
                'totalWorkedSuppNightHours' => $attendance['suppNight'], //input
                'totalWorkedSuppHolyDaysHours' => $attendance['suppHoly'], //input
                'totalWorkedweekendHours' => $attendance['weekend'], //input
                'totalWorkedHours' => $totalWorkedHours, //input
                'detailSuppl' => json_encode($detailSupp),
                'leaveDays' => $leaves, //input
                'absenceDaysUnJustified' => $absences, //input
                'absenceDaysJustified' => $absencesJustified, //input
                'totalAdvance' => $advances, //input
                // Family information
                // SMIG Compliance
                'smig_compliant' => $smigValidation['is_compliant'],
                'smig_minimum_required' => $smigValidation['minimum_required'],
                'smig_deficit' => $smigValidation['deficit_amount'],
                // Compensation
                'compensation' => $grossSalary ? json_encode($compensations) : null,
                'taxableCompensations' => $taxableCompensations,
                'nonTaxableCompensations' => $nonTaxableCompensations,
                // Salary amounts
                'detailGrossSalary' => json_encode($salaryCalculation),
                'grossSalary' => $grossSalary,
                'netSalary' => $grossSalary ?  $netSalary : 0,
                'netSalaryInWord' => $this->netSalaryToWords($netSalary),
                // 🔥 COMPLETE SOCIAL SECURITY (Employee + Employer)
                'cnssBase' => $cnssBase,
                // Employee deductions (for payroll)
                'cnss' => $socialSecurity['employee']['cnss']['amount'],
                'cnssRate' => $socialSecurity['employee']['cnss']['rate_percentage'],
                'amo' => $socialSecurity['employee']['amo']['amount'],
                'amoRate' => $socialSecurity['employee']['amo']['rate_percentage'],
                'lossJobCompensation' => $socialSecurity['employee']['ipe']['amount'],
                'lossJobRate' => $socialSecurity['employee']['ipe']['rate_percentage'],
                // Employer costs (for accounting)
                'employer_cnss' => $socialSecurity['employer']['cnss']['amount'],
                'employer_cnss_rate' => $socialSecurity['employer']['cnss']['rate_percentage'],
                'employer_amo' => $socialSecurity['employer']['amo']['amount'],
                'employer_amo_rate' => $socialSecurity['employer']['amo']['rate_percentage'],
                'employer_ipe' => $socialSecurity['employer']['ipe']['amount'],
                'employer_ipe_rate' => $socialSecurity['employer']['ipe']['rate_percentage'],
                'employer_tfp' => $socialSecurity['employer']['tfp']['amount'],
                'employer_tfp_rate' => $socialSecurity['employer']['tfp']['rate_percentage'],
                // 🔥 NEW: Family Allowance
                'employer_family_allowance' => $socialSecurity['employer']['family_allowance']['amount'],
                'employer_family_allowance_rate' => $socialSecurity['employer']['family_allowance']['rate_percentage'],
                'total_employer_cost' => $socialSecurity['employer']['total_amount'],
                // IR with rate
                'ir' => $irCalculation['amount'],
                'irRate' => $irCalculation['rate_percentage'],
                'detailIr' => json_encode($irCalculation),
                'tax_credit' => $irCalculation['tax_credit'],
                // Professional Fees with rate
                'proFees' => $proFeesCalculation['amount'],
                'proFeesRate' => $proFeesCalculation['rate_percentage'],
                // Retirement with rate
                'retirement' => $retirementCalculation['amount'],
                'retirementRate' => $retirementCalculation['rate_percentage'],

                // Totals
                'totalDeductions' => $grossSalary ? $totalEmployeeDeductions : 0,
                'total_social_security' => $socialSecurity['total_social_security'],
                // Metadata
                'generatedBy' => request()->user()->name ?? 'System',
                'calculation_year' => $year,
                'created_at' => now(),
                'updated_at' => now(),
            ];
        }

        if (empty($payrolls)) {
            return response()->json([
                'success' => false,
                'message' => 'Aucune fiche de paie n’a été ajoutée. Aucun employé éligible au traitement.'
            ], 400);
        }

        $request = new Request();
        $request->replace($payrolls);

        return $this->bulkStore($request);
    }


    function netSalaryToWords(float $netSalary): string
    {
        $euros = floor($netSalary); // integer part
        $cents = round(($netSalary - $euros) * 100); // decimal part

        $netSalaryInWord = SpellNumber::value($euros)
            ->locale('fr')
            ->currency('Dirhams')
            ->toMoney();

        if ($cents > 0) {
            $netSalaryInWord .= ' et ' . SpellNumber::value($cents)
                ->locale('fr')
                ->currency('centimes')
                ->toMoney();
        }

        return $netSalaryInWord;
    }

    // =========================================================
    // 🔹 Rate Calculation Methods
    // =========================================================

    // =========================================================
    // 🔹 Updated Calculation Methods with SMIG Floor
    // =========================================================

    // =========================================================
    // 🔹 Family Charge Calculation
    // =========================================================

    private function calculateFamilyCharge(string $maritalStatus, int $childrenCount): float
    {
        $maritalStatus = strtolower($maritalStatus);

        if ($maritalStatus === 'celibataire') {
            return 0;
        }

        if ($maritalStatus === 'marie' && $childrenCount === 0) {
            return self::FAMILY_CHARGE_MARRIED_NO_CHILDREN;
        }

        if ($childrenCount > 5) {
            return self::FAMILY_CHARGE_MAX;
        }

        return (self::FAMILY_CHARGE_PER_CHILD * $childrenCount) + self::FAMILY_CHARGE_MARRIED_NO_CHILDREN;
    }

    private function getEmployeeFamilyData(int $employeeId): array
    {
        $employee = Employee::select(['maritalStatus', 'childrenNumber'])
            ->find($employeeId);

        $maritalStatus = $employee->maritalStatus ?? 'celibataire';
        $childrenCount = (int) ($employee->childrenNumber ?? 0);
        $familyCharge = $this->calculateFamilyCharge($maritalStatus, $childrenCount);

        return [
            'marital_status' => $maritalStatus,
            'children_count' => $childrenCount,
            'family_charge' => $familyCharge,
        ];
    }


    private function calculateCNSSWithRate(float $base): array
    {
        // Apply both floor and ceiling
        $cnssBase = max(min($base, self::CNSS_CAP), self::CNSS_FLOOR);
        $rate = self::CNSS_RATE;
        $amount = $cnssBase * $rate;

        $is_capped = $base > self::CNSS_CAP;
        $is_floored = $base < self::CNSS_FLOOR;

        return [
            'amount' => $amount,
            'rate' => $rate,
            'rate_percentage' => $rate * 100,
            'rate_display' => (string) $rate * 100 . '%',
            'base' => $cnssBase,
            'is_capped' => $is_capped,
            'is_floored' => $is_floored,
            'ceiling_applied' => self::CNSS_CAP,
            'floor_applied' => self::CNSS_FLOOR,
            'year' => 2025
        ];
    }

    private function calculateAMOWithRate(float $base): array
    {
        // Apply both floor and ceiling
        $amoBase = max(min($base, self::AMO_CAP), self::AMO_FLOOR);
        $rate = self::AMO_RATE;
        $amount = $amoBase * $rate;

        $is_capped = $base > self::AMO_CAP;
        $is_floored = $base < self::AMO_FLOOR;

        return [
            'amount' => $amount,
            'rate' => $rate,
            'rate_percentage' => $rate * 100,
            'rate_display' => (string) $rate * 100 . '%',
            'base' => $amoBase,
            'is_capped' => $is_capped,
            'is_floored' => $is_floored,
            'ceiling_applied' => self::AMO_CAP,
            'floor_applied' => self::AMO_FLOOR,
            'year' => 2025
        ];
    }

    private function calculateLossJobWithRate(float $base): array
    {
        // Apply both floor and ceiling
        $lossJobBase = max(min($base, self::LOSS_JOB_CAP), self::LOSS_JOB_FLOOR);
        $rate = self::LOSS_JOB_RATE;
        $amount = $lossJobBase * $rate;

        $is_capped = $base > self::LOSS_JOB_CAP;
        $is_floored = $base < self::LOSS_JOB_FLOOR;

        return [
            'amount' => $amount,
            'rate' => $rate,
            'rate_percentage' => $rate * 100,
            'rate_display' => (string) $rate * 100 . '%',
            'base' => $lossJobBase,
            'is_capped' => $is_capped,
            'is_floored' => $is_floored,
            'ceiling_applied' => self::LOSS_JOB_CAP,
            'floor_applied' => self::LOSS_JOB_FLOOR,
            'year' => 2025
        ];
    }

    // =========================================================
    // 🔹 SMIG Validation Method
    // =========================================================

    private function validateSMIGCompliance(float $grossSalary, float $workedHours): array
    {
        $smigHourly = self::SMIG_HOURLY;
        $smigMonthly = self::SMIG_MONTHLY;
        $minimumRequired = ($workedHours * $smigHourly);

        $isCompliant = $grossSalary >= $minimumRequired;
        $deficit = max($minimumRequired - $grossSalary, 0);

        return [
            'is_compliant' => $isCompliant,
            'smig_hourly' => $smigHourly,
            'smig_monthly' => $smigMonthly,
            'worked_hours' => $workedHours,
            'minimum_required' => $minimumRequired,
            'actual_salary' => $grossSalary,
            'deficit_amount' => $deficit,
            'compliance_status' => $isCompliant ? 'Compliant' : 'Non-compliant'
        ];
    }

    private function calculateProFeesWithRate(float $gross): array
    {
        // Determine rate, cap, and type based on gross salary
        if ($gross <= self::PRO_FEES_THRESHOLD) {
            $rate = self::PRO_FEES_RATE_LOW;
            $cap = self::PRO_FEES_CAP_LOW;
            $rateType = 'low';
        } else {
            $rate = self::PRO_FEES_RATE_HIGH;
            $cap = self::PRO_FEES_CAP_HIGH;
            $rateType = 'high';
        }

        // Calculate amount with appropriate cap
        $amount = min($gross * $rate, $cap);

        // Return structured result
        return [
            'amount' => $amount,
            'rate' => $rate,
            'rate_percentage' => $rate * 100,
            'rate_display' => number_format($rate * 100, 2) . '%',
            'base' => $gross,
            'capped' => $amount >= $cap,
            'rate_type' => $rateType
        ];
    }


    private function calculateIRWithRate(float $grossSalary, float $totalDeductions, int $employeeId): array
    {

        // 1) Salaire imposable après toutes les déductions sociales
        $taxableIncome = round($grossSalary - $totalDeductions, 2);

        // 2) Trouver le bracket IR
        $bracket = null;
        $prevMax = 0.0;

        foreach (self::IR_BRACKETS as $b) {
            $max = $b['max'] ?? PHP_FLOAT_MAX;
            if ($taxableIncome > $prevMax && $taxableIncome <= $max) {
                $bracket = $b;
                break;
            }
            $prevMax = $max;
        }

        if (!$bracket) {
            // fallback, should never happen
            $bracket = ['rate' => 0, 'deduction' => 0];
        }

        $taxRate = $bracket['rate'];
        $deduction = $bracket['deduction'] ?? 0;

        // 3) IR brut avant deduction du bracket
        $amountRaw = round(max($taxableIncome * $taxRate, 0), 2);

        // 4) IR après deduction du bracket mais avant tax credit
        $amountBeforeDependents = round(max($amountRaw - $deduction, 0), 2);

        // 5) Nombre de dependants
        $dependentsCount = $this->getDependentsCount($employeeId);

        // 6) Tax credit limité
        $taxCredit = min(self::TAX_CREDIT_PER_DEPENDENT * $dependentsCount, self::TAX_CREDIT_MAX);
        if ($bracket['rate'] === 0) {
            $taxCredit = 0;
        }

        // 7) IR net final après tax credit
        $amount = max($amountBeforeDependents - $taxCredit, 0);

        // 8) Retour
        return [
            'amount'                     => $amount,                   // IR final après tax credit
            'rate'                       => $taxRate,
            'amount_raw'                 => $amountRaw,               // IR brut avant bracket
            'amount_before_dependents'   => $amountBeforeDependents, // IR après bracket, avant tax credit
            'rate_percentage'            => $taxRate * 100,
            'rate_display'               => $taxRate * 100 . '%',
            'taxable_income'             => $taxableIncome,
            'tax_credit'                 => $taxCredit,
            'dependents_count'           => $dependentsCount
        ];
    }


    /**
     * Count dependents eligible for tax credit (children + spouse as one).
     */
    private function getDependentsCount(int $employeeId): int
    {
        return (int) Employee::whereKey($employeeId)->value('relatedPerson') ?? 0;
    }


    // =========================================================
    // 🔹 Helper Methods (Keep existing ones)
    // =========================================================

    private function getAttendanceSummary(
        int $employeeId,
        int $month,
        int $year
    ): array {
        $data = Attendance::query()
            ->where('employee_id', $employeeId)
            ->whereMonth('date', $month)
            ->whereYear('date', $year)
            ->selectRaw('
            COALESCE(SUM(workedRegularHours), 0)
            + COALESCE(SUM(CASE WHEN type = "bankholiday" THEN 8 ELSE 0 END), 0)
            AS workedRegular,

            COALESCE(SUM(workedSuppDayHours), 0) as suppDay,
            COALESCE(SUM(workedSuppNightHours), 0) as suppNight,
            COALESCE(SUM(workedSuppHolyDaysHours), 0) as suppHoly,
            COALESCE(SUM(workedWeekendHours), 0) as weekend
        ')
            ->first();

        return [
            'workedRegular' => (float) $data->workedRegular,
            'suppDay' => (float) $data->suppDay,
            'suppNight' => (float) $data->suppNight,
            'suppHoly' => (float) $data->suppHoly,
            'weekend' => (float) $data->weekend,
        ];
    }


    private function getUnjustifiedAbsenceDays(
        int $employeeId,
        int $month,
        int $year
    ): int {
        return Absence::query()
            ->where('employee_id', $employeeId)
            ->whereMonth('date', $month)
            ->whereYear('date', $year)
            ->where('status', 'unjustified')
            ->count();
    }

    private function getAbsencesJustified(int $employeeId, int $month, int $year): float
    {
        return Absence::query()
            ->where('employee_id', $employeeId)
            ->whereMonth('date', $month)
            ->whereYear('date', $year)
            ->where('status', 'justified')
            ->count();
    }

    private function getLeaves(
        int $employeeId,
        int $month,
        int $year
    ): float {
        $monthStart = Carbon::create($year, $month, 1)->startOfMonth();
        $monthEnd   = $monthStart->copy()->endOfMonth();

        return (float) Leave::query()
            ->where('employee_id', $employeeId)
            ->whereIn('status', [
                Leave::STATUS_ONGOING,
                Leave::STATUS_TAKEN,
            ])
            // Overlapping range logic
            ->whereDate('start', '<=', $monthEnd)
            ->whereDate('end', '>=', $monthStart)
            ->get()
            ->sum('countedDays');
    }


    private function getAdvances(int $employeeId, int $month, int $year): float
    {
        return DB::transaction(function () use ($employeeId, $month, $year) {

            $advances = Advance::query()
                ->where('employee_id', $employeeId)
                ->whereYear('dateDeduction', $year)
                ->whereMonth('dateDeduction', $month)
                ->lockForUpdate() // 🔒 avoid double consumption
                ->where('status', 'taken')
                ->get();

            $total = (float) $advances->sum('amount');

            Advance::whereIn('id', $advances->pluck('id'))
                ->update(['status' => 'deduct', 'replay' => 'Deduction de la paie :' . $month . '/' . $year]);

            return $total;
        });
    }


    private function getCompensations(int $employeeId, int $month, int $year): array
    {
        return DB::transaction(function () use ($employeeId, $month, $year) {

            // Get compensations that are approved and not yet processed
            $compensations = Compensation::query()
                ->whereJsonContains('employee->id', $employeeId)
                ->whereMonth('dueDate', $month)
                ->whereYear('dueDate', $year)
                ->where('status', 'approved')
                ->lockForUpdate() // 🔒 avoid double consumption
                ->get();

            $compensationDetails = [];
            $taxableTotal = 0;
            $nonTaxableTotal = 0;
            $cnssBaseTotal = 0;

            // Collect compensation IDs for batch update
            $compensationIds = [];

            foreach ($compensations as $comp) {

                $allowanceData = is_string($comp->allowance)
                    ? json_decode($comp->allowance, true)
                    : $comp->allowance;

                if (empty($allowanceData)) {
                    continue;
                }

                // Normalize single allowance
                if (isset($allowanceData['amount'])) {
                    $allowanceData = [$allowanceData];
                }

                $hasValidAllowances = false;

                foreach ($allowanceData as $a) {
                    if (!isset($a['amount'])) {
                        continue;
                    }

                    $amount = (float) $a['amount'];
                    $label  = $a['name'] ?? 'Unnamed';

                    $isTaxable            = $a['taxable'] ?? true;
                    $isCnssContributable  = $a['isCnssContributable'] ?? true;
                    $exemptionLimit       = (float) ($a['exemptionLimit'] ?? 0);
                    $taxableAmount        = (float) ($a['taxableAmount'] ?? $amount);
                    $cnssBaseAmount       = (float) ($a['cnssBaseAmount'] ?? $amount);

                    $finalTaxableAmount = $this->applyExemptionLimit(
                        $label,
                        $amount,
                        $taxableAmount,
                        $exemptionLimit
                    );

                    $finalCnssBaseAmount = $isCnssContributable ? $cnssBaseAmount : 0;

                    $compensationDetails[] = [
                        'compensation_name' => $comp->name ?? $label,
                        'allowance_label'   => $label,
                        'amount'            => $amount,
                        'taxable'           => $isTaxable,
                        'taxable_amount'    => $finalTaxableAmount,
                        'cnss_base_amount'  => $finalCnssBaseAmount,
                        'dueDate'           => $comp->dueDate,
                        'compensation_id'   => $comp->id,
                    ];

                    if ($isTaxable) {
                        $taxableTotal += $finalTaxableAmount;
                        $nonTaxableTotal += ($amount - $finalTaxableAmount);
                    } else {
                        $nonTaxableTotal += $amount;
                    }

                    $cnssBaseTotal += $finalCnssBaseAmount;

                    $hasValidAllowances = true;
                }

                // Add to IDs list if it had valid allowances
                if ($hasValidAllowances) {
                    $compensationIds[] = $comp->id;
                }
            }

            // ✅ Update status for all processed compensations (similar to advances)
            if (!empty($compensationIds)) {
                Compensation::whereIn('id', $compensationIds)
                    ->update([
                        'status' => 'added',
                    ]);
            }

            /* ===============================
         * Seniority Allowance
         * =============================== */

            $seniorityAllowance = $this->calculateSeniorityAllowance($employeeId, $month, $year);

            if ($seniorityAllowance['amount'] > 0) {

                $amount = $seniorityAllowance['amount'];

                $compensationDetails[] = [
                    'compensation_name' => "Indemnité d'ancienneté",
                    'allowance_label'   => "Indemnité d'ancienneté",
                    'base'              => $seniorityAllowance['base'],
                    'amount'            => $amount,
                    'taxable'           => $seniorityAllowance['is_taxable'],
                    'taxable_amount'    => $amount,
                    'cnss_base_amount'  => $seniorityAllowance['is_cnss_contributable'] ? $amount : 0,
                    'dueDate'           => Carbon::create($year, $month, 25)->toDateString(),
                    'rate_percentage'   => $seniorityAllowance['rate_percentage'],
                ];

                if ($seniorityAllowance['is_taxable']) {
                    $taxableTotal += $amount;
                } else {
                    $nonTaxableTotal += $amount;
                }

                $cnssBaseTotal += $seniorityAllowance['is_cnss_contributable'] ? $amount : 0;
            }

            return [
                'details'           => $compensationDetails,
                'taxable_total'     => $taxableTotal,
                'non_taxable_total' => $nonTaxableTotal,
                'cnss_base_total'   => $cnssBaseTotal,
                'total'             => $taxableTotal + $nonTaxableTotal,
                'processed_compensation_ids' => $compensationIds,
            ];
        });
    }



    // =========================================================
    // 🔹 Seniority Allowance Calculation
    // =========================================================

    private function calculateSeniorityAllowance(int $employeeId, int $month, int $year): array
    {
        $employee = Employee::select('salary', 'hireDate')
            ->find($employeeId);

        if (!$employee) {
            return [
                'amount' => 0,
                'is_taxable' => true,
                'is_cnss_contributable' => true,
                'rate_percentage' => 0.0
            ];
        }

        $hireDate = $employee->hireDate ?? null;

        if (!$hireDate) {
            return [
                'base' => $employee->salary,
                'amount' => 0,
                'is_taxable' => true,
                'is_cnss_contributable' => true,
                'rate_percentage' => 0.0
            ];
        }

        $currentDate = Carbon::create($year, $month, 1)->endOfMonth();
        $hireDate = Carbon::parse($hireDate);

        $yearsOfService = $hireDate->diffInYears($currentDate);

        if ($yearsOfService < 2) {
            return [
                'base' => $employee->salary,
                'amount' => 0,
                'is_taxable' => true,
                'is_cnss_contributable' => true,
                'rate_percentage' => 0.0
            ];
        }

        $baseSalary = (float) ($employee->salary ?? 0);
        $seniorityPercentage = $this->getSeniorityPercentage($yearsOfService);
        $seniorityAmount = $baseSalary * ($seniorityPercentage / 100);

        return [
            'base' => $employee->salary,
            'amount' => $seniorityAmount,
            'is_taxable' => true,
            'is_cnss_contributable' => true,
            'rate_percentage' => $seniorityPercentage
        ];
    }

    private function getSeniorityPercentage(int $yearsOfService): float
    {
        if ($yearsOfService >= 25)
            return 25.0;
        if ($yearsOfService >= 20)
            return 20.0;
        if ($yearsOfService >= 12)
            return 15.0;
        if ($yearsOfService >= 5)
            return 10.0;
        if ($yearsOfService >= 2)
            return 5.0;
        return 0.0;
    }

    // =========================================================
    // 🔹 Gross Salary Calculation
    // =========================================================

    private function getBaseSalary(object $employee)
    {
        //set the base salary : if  $employee->salary the return it if not define or null then take hourePrice * 8
        // If salary exists and is not null → return it
        if (!empty($employee->salary)) {
            return (float) $employee->salary;
        }

        // Fallback: hour price × 8 hours
        $hourPrice = (float) ($employee->hourPrice ?? 0);
        $workingHours = (float) ($employee->workingHours ?? 0);
        return $hourPrice *  $workingHours;
    }

    private function calculateGrossSalary(
        object $employee,
        array $attendance,
        array $compensations,
        float $absenceUnjustified,
        float $absenceJustified,
        float $leaves,
        int $month,
        int $year
    ): array {
        // Base salary and work info
        $baseSalary = (float)($employee->salaryDeclared) ?? $this->getBaseSalary($employee);
        $workingHoursPerDay = (float) ($employee->workingHours ?? 8);

        // ✅ Default daysDeclared to 26 if it's missing or 0
        $daysDeclared = (float) ($employee->daysDeclared ?? 26);
        if ($daysDeclared <= 0) {
            $daysDeclared = 26;
        }

        // ✅ Use SMIG if salary is 0 or below SMIG
        if ($baseSalary <= 0) {
            $baseSalary = self::SMIG_MONTHLY;
        }

        //holydays from attendance
        $holydays = Attendance::where('employee_id', $employee->id)
            ->whereMonth('date', $month)
            ->whereYear('date', $year)
            ->where('type', 'holiday')
            ->count();

        // Convert worked hours to days
        $workedRegularHours = $attendance['workedRegular'] ?? 0;
        $actualWorkedDays = $workingHoursPerDay > 0 ? $workedRegularHours / $workingHoursPerDay : 0;
        //if this month has 5 sundays then add one day in the paid days
        // Convert worked hours → days
        $actualWorkedDaysNormal = $workingHoursPerDay > 0
            ? $workedRegularHours / $workingHoursPerDay
            : 0;
        $actualWorkedDays = $this->adjustWorkedDays($month, $year,  $actualWorkedDaysNormal);

        // Total paid days (real days)
        $totalPaidDays = $actualWorkedDays;
        // Daily rate
        $dailyRate = $baseSalary / $daysDeclared;

        // Base salary for period
        $baseSalaryForPeriod = $totalPaidDays * $dailyRate;

        // Deduction for unjustified absence
        $absenceDeduction = max($absenceUnjustified, 0) * $dailyRate;

        // Hourly rate for supplementary pay
        $hourlyRate = $workingHoursPerDay > 0 ? $dailyRate / $workingHoursPerDay : 0;

        // Supplementary pay (overtime)
        $supplementaryPay = 0;
        $supplementaryPay += ($attendance['suppDay'] ?? 0) * $hourlyRate * 1.25;
        $supplementaryPay += ($attendance['suppNight'] ?? 0) * $hourlyRate * 1.5;
        $supplementaryPay += ($attendance['suppHoly'] ?? 0) * $hourlyRate * 2.0;
        $supplementaryPay += ($attendance['weekend'] ?? 0) * $hourlyRate * 1.5;

        // Compensations
        $compensationTotal = ($compensations['taxable_total'] ?? 0) + ($compensations['non_taxable_total'] ?? 0);
        $taxableCompensations = $compensations['taxable_total'] ?? 0;
        $nonTaxableCompensations = $compensations['non_taxable_total'] ?? 0;
        $compensationCnssBase = $compensations['cnss_base_total'] ?? 0;
        $leavesCost = $leaves * $dailyRate;
        $absenceJustifiedCost = $absenceJustified * $dailyRate;
        // Correct gross salary
        $grossSalary = $baseSalaryForPeriod  + $supplementaryPay + $compensationTotal + $leavesCost + $absenceJustifiedCost;
        // CNSS base capped
        $cnssBase = min($grossSalary + $compensationCnssBase, self::CNSS_CAP);
        return [
            'gross_salary' => max($grossSalary, 0),
            'gross_salary_taxable' => $grossSalary - $nonTaxableCompensations,
            'cnss_base' => $cnssBase,
            'taxable_compensations' => $taxableCompensations,
            'non_taxable_compensations' => $nonTaxableCompensations,
            'base_salary' => $baseSalaryForPeriod,
            'supplementary_pay' => $supplementaryPay,
            'absence_deduction' => $absenceDeduction,
            'daily_rate' => $dailyRate,
            'hourly_rate' => $hourlyRate,
            'holydays' => $holydays,
            'total_paid_days' => $totalPaidDays
        ];
    }

    private function adjustWorkedDays(int $month, int $year, int $actualWorkedDays): int
    {
        if ($this->monthHasFiveSundays($month, $year)) {
            $actualWorkedDays += 1;
        }
        return $actualWorkedDays;
    }

    private function monthHasFiveSundays(int $month, int $year): bool
    {
        $sundays = 0;
        $date = Carbon::createFromDate($year, $month, 1);
        $endOfMonth = $date->copy()->endOfMonth();
        while ($date->lte($endOfMonth)) {
            if ($date->isSunday()) {
                $sundays++;
            }
            $date->addDay();
        }
        return $sundays >= 5;
    }


    private function countSundays(int $month, int $year): int
    {
        $sundays = 0;
        $daysInMonth = cal_days_in_month(CAL_GREGORIAN, $month, $year);

        for ($day = 1; $day <= $daysInMonth; $day++) {
            $date = Carbon::createFromDate($year, $month, $day);
            if ($date->dayOfWeek === Carbon::SUNDAY) {
                $sundays++;
            }
        }

        return $sundays;
    }

    private function applyExemptionLimit(string $label, float $amount, float $taxableAmount, float $exemptionLimit): float
    {
        if ($exemptionLimit > 0) {
            $taxable = max($amount - $exemptionLimit, 0);
            return min($taxable, $taxableAmount);
        }

        $lowerLabel = strtolower($label);

        if (str_contains($lowerLabel, 'transport')) {
            return max($amount - self::EXEMPTION_LIMITS['transport'], 0);
        }
        if (str_contains($lowerLabel, 'meal') || str_contains($lowerLabel, 'repas')) {
            return max($amount - self::EXEMPTION_LIMITS['meal'], 0);
        }
        if (str_contains($lowerLabel, 'family') || str_contains($lowerLabel, 'familial')) {
            return max($amount - self::EXEMPTION_LIMITS['family'], 0);
        }
        if (str_contains($lowerLabel, 'housing') || str_contains($lowerLabel, 'logement')) {
            return max($amount - (self::EXEMPTION_LIMITS['housing'] / 12), 0);
        }

        return $taxableAmount;
    }

    private function calculateHourlyRate(object $employee, float $dailyRate, float $workingHoursPerDay): float
    {
        $employeeHourPrice = (float) ($employee->hourPrice ?? 0);
        if ($employeeHourPrice > 0) {
            return $employeeHourPrice;
        }
        return $workingHoursPerDay > 0 ? $dailyRate / $workingHoursPerDay : 0;
    }

    private function calculateSupplementaryPay(array $attendance, float $hourlyRate): float
    {
        $supplementaryPay = 0;
        $supplementaryPay += ($attendance['suppDay'] ?? 0) * $hourlyRate * 1.25;
        $supplementaryPay += ($attendance['suppNight'] ?? 0) * $hourlyRate * 1.5;
        $supplementaryPay += ($attendance['suppHoly'] ?? 0) * $hourlyRate * 2.0;
        $supplementaryPay += ($attendance['weekend'] ?? 0) * $hourlyRate * 1.5;

        return $supplementaryPay;
    }

    private function detailSupplementaryPay(array $attendance, float $hourlyRate): array
    {
        $details = [];

        foreach (self::SUPPLEMENTARY_TYPES as $key => $info) {
            $hours = $attendance[$key] ?? 0;
            if ($hours > 0) {
                $amount = $hours * $hourlyRate * $info['rate'];
                $details[] = [
                    'label' => $info['label'],
                    'base_regulier' => $hourlyRate,
                    'nbrHour' => $hours,
                    'rateLabel' => $info['rateLabel'],
                    'rate' => $info['rate'],
                    'amount' => $amount,
                ];
            }
        }

        return $details;
    }



    // =========================================================
    // 🔹 Download Payslip Method (Fixed)
    // =========================================================

    public function downloadPayslip(Request $request)
    {
        try {
            $payrolls = Payroll::with('employee')
                ->whereIn('employee_id', $request->ids)
                ->where('month', $request->month)
                ->get();

            if ($payrolls->isEmpty()) {
                return response()->json([
                    'success' => false,
                    'error' => 'No payrolls found'
                ], 404);
            }

            // FIXED: Pass as 'payrolls' not 'payslip'
            $pdf = Pdf::setOptions([
                'isRemoteEnabled' => true,  // ✅ allow HTTPS images
                'isHtml5ParserEnabled' => true,
            ])
                ->loadView('payroll.payslip', ['payrolls' => $payrolls]);
            $pdf->setPaper('A4', 'portrait');

            return response($pdf->output(), 200)
                ->header('Content-Type', 'application/pdf')
                ->header('Content-Disposition', 'inline; filename="payslips.pdf"');
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Error generating payslip PDF: ' . $e->getMessage()
            ], 500);
        }
    }

    // =========================================================
    // 🔹 Complete Social Security Calculation (Employee + Employer)
    // =========================================================

    private function calculateCompleteSocialSecurity(float $base, float $bsi, int $childrenCount = 0): array
    {
        // Calculate employee contributions (for payroll deductions)
        $employeeCNSS = $this->calculateCNSSEmployee($base);
        $employeeAMO = $this->calculateAMOEmployee($bsi);
        $employeeIPE = $this->calculateIPEEmployee($bsi);

        // Calculate employer contributions (for company costs)
        $employerCNSS = $this->calculateCNSSEmployer($base);
        $employerAMO = $this->calculateAMOEmployer($bsi);
        $employerIPE = $this->calculateIPEEmployer($bsi);
        $employerTFP = $this->calculateTFPEmployer($bsi);
        $employerFamilyAllowance = $this->calculateFamilyAllowance($base, $childrenCount);

        // Totals
        $totalEmployee = $employeeCNSS['amount'] + $employeeAMO['amount'] + $employeeIPE['amount'];
        $totalEmployer = $employerCNSS['amount'] + $employerAMO['amount'] + $employerIPE['amount'] + $employerTFP['amount'] + $employerFamilyAllowance['amount'];
        $totalSocialSecurity = $totalEmployee + $totalEmployer;

        return [
            // Employee deductions
            'employee' => [
                'cnss' => $employeeCNSS,
                'amo' => $employeeAMO,
                'ipe' => $employeeIPE,
                'total_amount' => $totalEmployee,
                'total_percentage' => $base != 0
                    ? ($totalEmployee / $base) * 100
                    : 0,
            ],
            // Employer contributions
            'employer' => [
                'cnss' => $employerCNSS,
                'amo' => $employerAMO,
                'ipe' => $employerIPE,
                'tfp' => $employerTFP,
                'family_allowance' => $employerFamilyAllowance,
                'total_amount' => $totalEmployer,
                'total_percentage' => $base != 0
                    ? ($totalEmployee / $base) * 100
                    : 0
            ],
            // Grand totals
            'total_social_security' => $totalSocialSecurity,
            'total_percentage' => $base != 0
                ? ($totalEmployee / $base) * 100
                : 0,
            'base_salary' => $base,
            'children_count' => $childrenCount
        ];
    }

    // =========================================================
    // 🔹 Employee Calculation Methods
    // =========================================================

    private function calculateCNSSEmployee(float $base): array
    {
        $cnssBase = min($base, self::CNSS_CAP);
        $rate = self::CNSS_EMPLOYEE_RATE;
        $amount = $cnssBase * $rate;

        return [
            'amount' => $amount,
            'rate' => $rate,
            'rate_percentage' => $rate * 100,
            'base' => $cnssBase,
            'is_capped' => $base > self::CNSS_CAP,
            'type' => 'employee'
        ];
    }

    private function calculateAMOEmployee(float $base): array
    {
        $rate = self::AMO_EMPLOYEE_RATE;
        $amount = $base * $rate;

        return [
            'amount' => $amount,
            'rate' => $rate,
            'rate_percentage' => $rate * 100,
            'base' => $base,
            'is_capped' => false,
            'type' => 'employee'
        ];
    }

    private function calculateIPEEmployee(float $base): array
    {
        $ipeBase = min($base, self::IPE_CAP);
        $rate = self::IPE_EMPLOYEE_RATE;
        $amount = $ipeBase * $rate;

        return [
            'amount' => $amount,
            'rate' => $rate,
            'rate_percentage' => $rate * 100,
            'base' => $ipeBase,
            'is_capped' => $base > self::IPE_CAP,
            'type' => 'employee'
        ];
    }

    // =========================================================
    // 🔹 Employer Calculation Methods
    // =========================================================

    private function calculateCNSSEmployer(float $base): array
    {
        $cnssBase = min($base, self::CNSS_CAP);
        $rate = self::CNSS_EMPLOYER_RATE;
        $amount = $cnssBase * $rate;

        return [
            'amount' => $amount,
            'rate' => $rate,
            'rate_percentage' => $rate * 100,
            'base' => $cnssBase,
            'is_capped' => $base > self::CNSS_CAP,
            'type' => 'employer'
        ];
    }

    private function calculateAMOEmployer(float $base): array
    {
        $rate = self::AMO_EMPLOYER_RATE;
        $amount = $base * $rate;

        return [
            'amount' => $amount,
            'rate' => $rate,
            'rate_percentage' => $rate * 100,
            'base' => $base,
            'is_capped' => false,
            'type' => 'employer'
        ];
    }

    private function calculateIPEEmployer(float $base): array
    {
        $ipeBase = min($base, self::IPE_CAP);
        $rate = self::IPE_EMPLOYER_RATE;
        $amount = $ipeBase * $rate;

        return [
            'amount' => $amount,
            'rate' => $rate,
            'rate_percentage' => $rate * 100,
            'base' => $ipeBase,
            'is_capped' => $base > self::IPE_CAP,
            'type' => 'employer'
        ];
    }

    private function calculateTFPEmployer(float $base): array
    {
        $rate = self::TFP_EMPLOYER_RATE;
        $amount = $base * $rate;

        return [
            'amount' => $amount,
            'rate' => $rate,
            'rate_percentage' => $rate * 100,
            'base' => $base,
            'is_capped' => false,
            'type' => 'employer'
        ];
    }

    // =========================================================
    // 🔹 Family Allowance Calculation Method
    // =========================================================

    private function calculateFamilyAllowance(float $base, int $childrenCount): array
    {
        $familyBase = min($base, self::FAMILY_ALLOWANCE_CAP);
        $rate = self::FAMILY_ALLOWANCE_RATE;
        $amount =  $familyBase * $rate;

        return [
            'amount' => $amount,
            'rate' => $rate,
            'rate_percentage' =>  $rate * 100,  // 6.40
            'base' => $familyBase,
            'children_count' => $childrenCount,
            'is_capped' => $base > self::FAMILY_ALLOWANCE_CAP,
            'type' => 'employer',
            'description' => 'Allocations familiales'
        ];
    }

    // =========================================================
    // 🔹 Retirement Calculation Methods
    // =========================================================

    private function calculateRetirementWithRate(int $employeeId, float $currentSalary): array
    {
        // Get employee insurance days and salary history
        $employeeData = $this->getEmployeeRetirementData($employeeId);
        $insuranceDays = $employeeData['insurance_days'];
        $averageSalary = $employeeData['average_salary'] ?? $currentSalary;

        // Calculate retirement rate based on insurance days
        $retirementRate = $this->calculateRetirementRate($insuranceDays);

        // Calculate retirement amount
        $retirementAmount = $averageSalary * $retirementRate;

        return [
            'amount' => $retirementAmount,
            'rate' => $retirementRate,
            'rate_percentage' => $retirementRate * 100,  // 50.00, 55.00, etc.
            'rate_display' => (string) $retirementRate * 100 . '%',
            'insurance_days' => $insuranceDays,
            'average_salary' => $averageSalary,
            'calculation_details' => $this->getRetirementCalculationDetails($insuranceDays, $averageSalary, $retirementRate)
        ];
    }

    private function calculateRetirementRate(int $insuranceDays): float
    {
        // Check minimum requirement
        if ($insuranceDays < self::RETIREMENT_MIN_DAYS) {
            return 0.0;
        }

        // Calculate base rate
        $rate = self::RETIREMENT_BASE_RATE;

        // Calculate bonus for extra days
        $extraDays = $insuranceDays - self::RETIREMENT_MIN_DAYS;
        if ($extraDays > 0) {
            $bonusTranches = floor($extraDays / self::RETIREMENT_BONUS_DAYS);
            $bonusRate = $bonusTranches * self::RETIREMENT_BONUS_RATE;
            $rate += $bonusRate;
        }

        // Apply maximum rate cap
        return min($rate, self::RETIREMENT_MAX_RATE);
    }

    private function getEmployeeRetirementData(int $employeeId): array
    {
        $employee = Employee::query()
            ->select(['insuranceDays', 'hireDate'])
            ->find($employeeId);

        if (!$employee) {
            return [
                'insurance_days' => 0,
                'average_salary' => 0,
                'hire_date' => null,
            ];
        }

        return [
            'insurance_days' => (int) ($employee->insuranceDays ?? 0),
            'average_salary' => $this->calculateAverageSalary($employeeId),
            'hire_date' => $employee->hireDate ?? null,
        ];
    }


    private function calculateAverageSalary(int $employeeId): float
    {
        $salaryHistory = Payroll::query()
            ->where('employee_id', $employeeId)
            ->where('status', 'approved')
            ->orderByDesc('month')
            ->limit(self::RETIREMENT_AVERAGE_MONTHS)
            ->pluck('grossSalary')
            ->toArray();

        if (empty($salaryHistory)) {
            $salary = Employee::where('id', $employeeId)->value('salary');
            return (float) ($salary ?? 0);
        }

        return array_sum($salaryHistory) / count($salaryHistory);
    }


    private function getRetirementCalculationDetails(int $insuranceDays, float $averageSalary, float $rate): string
    {
        if ($insuranceDays < self::RETIREMENT_MIN_DAYS) {
            return "Insuffisant: {$insuranceDays}/" . self::RETIREMENT_MIN_DAYS . ' jours';
        }

        $baseRate = self::RETIREMENT_BASE_RATE * 100;
        $extraDays = $insuranceDays - self::RETIREMENT_MIN_DAYS;

        if ($extraDays > 0) {
            $bonusTranches = floor($extraDays / self::RETIREMENT_BONUS_DAYS);
            $bonusRate = $bonusTranches * self::RETIREMENT_BONUS_RATE * 100;
            return "{$baseRate}% base + {$bonusRate}% bonus ({$bonusTranches} tranches) = " . ($rate * 100) . '%';
        }

        return "{$baseRate}% base rate";
    }




    // all rest  crud  methode

    public function index(Request $request)
    {
        return $this->getData($request);
    }

    public function update(Request $request)
    {
        return $this->updateData($request);
    }

    public function bulkUpdate(Request $request)
    {
        return $this->multiUpdate($request);
    }

    public function destroy(Request $request)
    {
        return $this->destroyData($request);
    }

    public function bulkDestory(Request $request)
    {
        return $this->bulkDelete($request);
    }

    public function clone(Request $request)
    {
        return $this->cloneData($request);
    }

    public function export(Request $request)
    {
        return $this->exportData($request);
    }

    public function import(Request $request)
    {
        return $this->importData($request);
    }

    public function upload(Request $request)
    {
        return $this->uploadFile($request);
    }

    public function download(Request $request)
    {
        return $this->downloadPayslip($request);
    }
}
